import os
import Live
from typing import Tuple
from .handler import GripHandler
from .version import VERSION

class ApplicationHandler(GripHandler):
    def init_api(self):
        #--------------------------------------------------------------------------------
        # Generic callbacks
        #--------------------------------------------------------------------------------
        def get_version(_) -> Tuple:
            application = Live.Application.get_application()
            return application.get_major_version(), application.get_minor_version()
        self.osc_server.add_handler("/live/application/get/version", get_version)

        def get_process_id(_) -> Tuple:
            """Returns the Unix PID of this Ableton Live process."""
            return os.getpid(),
        self.osc_server.add_handler("/live/application/get/process_id", get_process_id)

        # Send startup signal (with version) and PID to daemon
        self.osc_server.send("/live/startup", (VERSION,))
        self.osc_server.send("/live/application/get/process_id", (os.getpid(),))

        def get_average_process_usage(_) -> Tuple:
            application = Live.Application.get_application()
            return application.average_process_usage,
        self.osc_server.add_handler("/live/application/get/average_process_usage", get_average_process_usage)
        self.osc_server.send("/live/application/get/average_process_usage")

        #--------------------------------------------------------------------------------
        # Introspection - discover Application properties
        #--------------------------------------------------------------------------------
        def introspect_application(_) -> Tuple:
            """
            Returns all property names available on Live.Application.get_application().
            Useful for discovering available properties like view.
            """
            application = Live.Application.get_application()
            properties = []
            
            # Get all attributes
            for attr_name in dir(application):
                # Skip private attributes and methods
                if attr_name.startswith('_'):
                    continue
                
                attr = getattr(application, attr_name)
                # Check if it's a property or non-callable attribute
                if not callable(attr):
                    try:
                        # Try to access it to see if it's readable
                        _ = getattr(application, attr_name)
                        properties.append(attr_name)
                    except:
                        pass
            
            return tuple(sorted(properties))
        
        self.osc_server.add_handler("/live/application/introspect", introspect_application)
